"""Calculates the Frechet Inception Distance (FID) to evalulate GANs

The FID metric calculates the distance between two distributions of images.
Typically, we have summary statistics (mean & covariance matrix) of one
of these distributions, while the 2nd distribution is given by a GAN.

When run as a stand-alone program, it compares the distribution of
images that are stored as PNG/JPEG at a specified location with a
distribution given by summary statistics (in pickle format).

The FID is calculated by assuming that X_1 and X_2 are the activations of
the pool_3 layer of the inception net for generated samples and real world
samples respectively.

See --help to see further details.

Code apapted from https://github.com/bioinf-jku/TTUR to use PyTorch instead
of Tensorflow

Copyright 2018 Institute of Bioinformatics, JKU Linz

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

   http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
"""
import ipdb
import os
from pathlib import Path
from argparse import ArgumentDefaultsHelpFormatter, ArgumentParser

import math

import clip

from pdb import set_trace as st
import json
import numpy as np
import torch
import torchvision.transforms as TF
from PIL import Image
from scipy import linalg
from torch.nn.functional import adaptive_avg_pool2d
import cv2
from tqdm import tqdm

from pytorch_fid.inception import InceptionV3

parser = ArgumentParser(formatter_class=ArgumentDefaultsHelpFormatter)
parser.add_argument('--batch-size', type=int, default=100,
                    help='Batch size to use')
parser.add_argument('--reso', type=int, default=128,
                    help='Batch size to use')
parser.add_argument('--num-workers', type=int, default=8,
                    help=('Number of processes to use for data loading. '
                          'Defaults to `min(8, num_cpus)`'))
parser.add_argument('--device', type=str, default=None,
                    help='Device to use. Like cuda, cuda:0 or cpu')
parser.add_argument('--dataset', type=str, default='omni',
                    help='Device to use. Like cuda, cuda:0 or cpu')
parser.add_argument('--dims', type=int, default=2048,
                    choices=list(InceptionV3.BLOCK_INDEX_BY_DIM),
                    help=('Dimensionality of Inception features to use. '
                          'By default, uses pool3 features'))
parser.add_argument('--save-stats', action='store_true',
                    help=('Generate an npz archive from a directory of samples. '
                          'The first path is used as input and the second as output.'))
parser.add_argument('path', type=str, nargs=2,
                    help=('Paths to the generated images or '
                          'to .npz statistic files'))

IMAGE_EXTENSIONS = {'bmp', 'jpg', 'jpeg', 'pgm', 'png', 'ppm',
                    'tif', 'tiff', 'webp'}



def compute_statistics_of_path(path, clip_model, preprocess, device, **kwargs):
    # basepath="/mnt/sfs-common/yslan/Repo/3dgen/FID-KID-Outputdir-objv/metrics/fid/gso_gt"
    # basepath="/mnt/sfs-common/yslan/Repo/3dgen/FID-KID-Outputdir-objv/metrics/fid-withtop/gso_gt"

    # basepath="/mnt/sfs-common/yslan/Repo/3dgen/FID-KID-Outputdir-free3d/metrics/fid-withtop/gso_gt"
    # basepath="/mnt/sfs-common/yslan/Repo/3dgen/FID-KID-Outputdir-/metrics/fid-withtop/gso_gt"

    basepath = '/mnt/sfs-common/yslan/Repo/3dgen/FID-KID-Outputdir-objv/clip_score'
    os.makedirs(os.path.join(basepath), exist_ok=True)

    save_path = os.path.join(basepath, 'gt.pt')

    if os.path.exists(save_path):
        clip_i_feat_dict = torch.load(save_path)
        return clip_i_feat_dict 

    objv_dataset = '/mnt/sfs-common/yslan/Dataset/Obajverse/chunk-jpeg-normal/bs_16_fixsave3/170K/512/'
    dataset_json = os.path.join(objv_dataset, 'dataset.json')
    with open(dataset_json, 'r') as f:
        dataset_json = json.load(f)

    # all_objs = dataset_json['Animals'][::3][:6250]
    all_objs = dataset_json['Animals'][::3][1100:2200]
    all_objs = all_objs[:600][:]

    # ! get CLIP-I features and return a dict
    data_dir="/mnt/sfs-common/yslan/Dataset/Obajverse/Objv-animals-for-FID_eval/" # hard-coded now.
    clip_i_feat_dict = {}

    for obj_folder in tqdm(all_objs[:]):
        for idx in [0]: # only query frontal view is enough
            image_path = os.path.join(data_dir, obj_folder, f'{idx}.jpg')
            image_input = preprocess(Image.open(image_path)).unsqueeze(0).to(device)
            # Calculate features
            with torch.no_grad():
                image_features = clip_model.encode_image(image_input)
                image_features /= image_features.norm(dim=-1, keepdim=True) # for calculating similarity
                clip_i_feat_dict[obj_folder] = image_features.cpu()
    
    torch.save(clip_i_feat_dict, save_path)

    return clip_i_feat_dict


def compute_statistics_of_path_new(path, clip_model, preprocess, device,):

    # basepath='/mnt/sfs-common/yslan/Repo/3dgen/FID-KID-Outputdir-objv/metrics/fid/'+str(reso)+dataset

    objv_dataset = '/mnt/sfs-common/yslan/Dataset/Obajverse/chunk-jpeg-normal/bs_16_fixsave3/170K/512/'
    dataset_json = os.path.join(objv_dataset, 'dataset.json')
    with open(dataset_json, 'r') as f:
        dataset_json = json.load(f)

    all_objs = dataset_json['Animals'][::3][1100:2200]
    all_objs = all_objs[:600]

    sample_name=path.split('/')[-1]

    basepath = '/mnt/sfs-common/yslan/Repo/3dgen/FID-KID-Outputdir-objv/clip_score/'
    sample_name = Path(path).relative_to('/mnt/sfs-common/yslan/Repo/3dgen/FID-KID-Outputdir-objv/')

    basepath = os.path.join(basepath, sample_name)
    os.makedirs(basepath, exist_ok=True)

    save_path = os.path.join(basepath, 'clip_score.pt') # 

    if os.path.exists(save_path):
        clip_i_feat_dict = torch.load(save_path)
        return clip_i_feat_dict, sample_name

    clip_i_feat_dict = {}

    for obj_folder in tqdm(all_objs):
        obj_folder = '/'.join(obj_folder.split('/')[1:])
        files = []
        for idx in range(24):
            # TODO

            # files.append(os.path.join(path, obj_folder, f'{idx}.jpg'))
            if 'Lara' in path:
                files.append(os.path.join(path, '/'.join(obj_folder.split('/')[:-1]), '0.jpg', f'{idx}.jpg'))
            elif 'GA' in path:
                files.append(os.path.join(path, '/'.join(obj_folder.split('/')[:-1]), '0', f'sample-0-{idx}.jpg'))
            elif 'scale3d' in path:
                files.append(os.path.join(path, '/'.join(obj_folder.split('/')[:-1]), '1', f'{idx}.png'))
            elif 'LRM' in path:
                files.append(os.path.join(path, '/'.join(obj_folder.split('/')[:-1]), '0', f'{idx}.jpg'))
            else:
                files.append(os.path.join(path, obj_folder, '0', f'{idx}.jpg'))
        # ! query files and record

        clip_i_feat_dict[obj_folder] = []
        for image_path in files:
            try:
                image_input = preprocess(Image.open(image_path)).unsqueeze(0).to(device)
                # Calculate features
                with torch.no_grad():
                    image_features = clip_model.encode_image(image_input)
                    image_features /= image_features.norm(dim=-1, keepdim=True) # for calculating similarity
                    clip_i_feat_dict[obj_folder].append(image_features.cpu())
            except:
                pass

    torch.save(clip_i_feat_dict, save_path)

    return clip_i_feat_dict, sample_name

def aggregate_clip_I_score(gt_clip_features, pred_clip_features):

    
    all_ins = []
    ins_similarity = {}
    for ins, gt_clip_feats in gt_clip_features.items():
        ins = '/'.join(ins.split('/')[1:])

        if ins not in pred_clip_features:
            continue

        ins_similarity[ins] = []
        for pred_frame_clip_feats in pred_clip_features[ins]:
            similarity = (torch.nn.functional.cosine_similarity(gt_clip_feats, pred_frame_clip_feats, dim=1, eps=1e-8) ).item() * 100
            ins_similarity[ins].append(similarity)

        if len(ins_similarity[ins]) == 0:
            continue
        all_ins.append(np.mean(ins_similarity[ins]))

    return np.mean(all_ins), ins_similarity



def calculate_clip_I_score_given_paths(paths, batch_size, device):
    """Calculates the FID of two paths"""


    model, preprocess = clip.load("ViT-B/32", device=device) # ! other model available
    gt_clip_features = compute_statistics_of_path(paths[0], model, preprocess, device)
    pred_clip_features, sample_name = compute_statistics_of_path_new(paths[1], model, preprocess, device)

    all_ins_clip_score, ins_similarity = aggregate_clip_I_score(gt_clip_features, pred_clip_features)

    basepath = '/mnt/sfs-common/yslan/Repo/3dgen/FID-KID-Outputdir-objv/clip_score/'
    torch.save(ins_similarity, os.path.join(basepath, sample_name, 'ins_similarity.pt'))

    return all_ins_clip_score


def main():
    args = parser.parse_args()

    if args.device is None:
        device = torch.device('cuda' if (torch.cuda.is_available()) else 'cpu')
    else:
        device = torch.device(args.device)

    clip_score = calculate_clip_I_score_given_paths(args.path,
                                          args.batch_size,
                                          device,)
    print(f'{args.dataset} clip_score: ', clip_score)


if __name__ == '__main__':
    main()
